// dashboard_signals.h
#pragma once
#include <stdint.h>
#include <stddef.h>
#include <stdio.h>   // snprintf

#ifdef __cplusplus
extern "C" {
#endif

// ----------------------- Mapping enum -----------------------
#define DASHBOARD_MAPPING(name, off, scale) MAP_##name,
#define DASHBOARD_SIGNAL(id, label, map)    /* no-op */
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */
typedef enum {
#include "dashboard_signals.def"
    DASHBOARD_MAPPING_COUNT
} DashboardMappingId;
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR

// ----------------------- Signal enum ------------------------
#define DASHBOARD_MAPPING(name, off, scale) /* no-op */
#define DASHBOARD_SIGNAL(id, label, map)    id,
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */
typedef enum {
#include "dashboard_signals.def"
    DASHBOARD_SIGNAL_COUNT
} DashboardSignalId;
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR

// ----------------------- Mapping table ----------------------
typedef struct {
    double offset;
    double scale;
    const char* name;
} DashboardMapping;

static const DashboardMapping gDashboardMappings[DASHBOARD_MAPPING_COUNT] = {
#define DASHBOARD_MAPPING(name, off, scale) [MAP_##name] = { (off), (scale), #name },
#define DASHBOARD_SIGNAL(id, label, map)    /* no-op */
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// ----------------------- Signal → name -----------------------
static const char* const gSignalEnumNames[DASHBOARD_SIGNAL_COUNT] = {
#define DASHBOARD_MAPPING(name, off, scale) /* no-op */
#define DASHBOARD_SIGNAL(id, label, map)    [id] = #id,
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// ----------------------- Signal → display --------------------
static const char* const gSignalDisplayNames[DASHBOARD_SIGNAL_COUNT] = {
#define DASHBOARD_MAPPING(name, off, scale) /* no-op */
#define DASHBOARD_SIGNAL(id, label, map)    [id] = label,
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// ----------------------- Signal → mapping --------------------
static const DashboardMappingId gSignalToMapping[DASHBOARD_SIGNAL_COUNT] = {
#define DASHBOARD_MAPPING(name, off, scale) /* no-op */
#define DASHBOARD_SIGNAL(id, label, map)    [id] = MAP_##map,
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// ----------------------- Helpers -----------------------------

__attribute__((weak))
void dashboard_write_bytes(const uint8_t* data, size_t len) {
    // default to stdout; override in your project
    fwrite(data, 1, len, stdout);
}

// Converts raw integer value to double by applying the appropriate mapping
static inline double dashboard_apply_mapping(DashboardSignalId id, int raw)
{
    DashboardMappingId mid = gSignalToMapping[id];
    const DashboardMapping* m = &gDashboardMappings[mid];
    return m->offset + m->scale * (double)raw;
}

// Writes formatted output line to serial debug stream
static inline void dashboard_emit_signal(DashboardSignalId id, int rawValue)
{
    char buf[64];
    int n = snprintf(buf, sizeof buf, "%s=%d\r\n", gSignalEnumNames[id], rawValue);
    if (n > 0) dashboard_write_bytes((const uint8_t*)buf, (size_t)n);
}


// --------------------- STATES: group enum ---------------------
#define DASHBOARD_MAPPING(name,off,scale)      /* no-op */
#define DASHBOARD_SIGNAL(id,label,map)         /* no-op */
#define DASHBOARD_STATE_GROUP(group, glabel)   ST_GROUP_##group,
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */

typedef enum {
#include "dashboard_signals.def"
    ST_GROUP_COUNT
} DashboardStateGroup;

#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR

// Group enum name and display strings
static const char* const gStateGroupNames[ST_GROUP_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)   [ST_GROUP_##group] = #group,
#define DASHBOARD_STATE(group, sid, slabel)
#define DASHBOARD_ERROR(id, code, msg)
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

static const char* const gStateGroupDisplay[ST_GROUP_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)   [ST_GROUP_##group] = glabel,
#define DASHBOARD_STATE(group, sid, slabel)
#define DASHBOARD_ERROR(id, code, msg)
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// --------------------- STATES: flat state enum ----------------
#define DASHBOARD_MAPPING(name,off,scale)      /* no-op */
#define DASHBOARD_SIGNAL(id,label,map)         /* no-op */
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    ST_##group##_##sid,
#define DASHBOARD_ERROR(id, code, msg)         /* no-op */

typedef enum {
#include "dashboard_signals.def"
    DASHBOARD_STATE_COUNT
} DashboardStateId;

#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR

// State → group (e.g., ST_DRIVE_GO → ST_GROUP_DRIVE)
static const DashboardStateGroup gStateToGroup[DASHBOARD_STATE_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)
#define DASHBOARD_STATE(group, sid, slabel)    [ST_##group##_##sid] = ST_GROUP_##group,
#define DASHBOARD_ERROR(id, code, msg)
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// State identifier names ("STOP", "GO", …) and display labels ("Stop", "Go", …)
static const char* const gStateNames[DASHBOARD_STATE_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)
#define DASHBOARD_STATE(group, sid, slabel)    [ST_##group##_##sid] = #sid,
#define DASHBOARD_ERROR(id, code, msg)
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

static const char* const gStateDisplay[DASHBOARD_STATE_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)
#define DASHBOARD_STATE(group, sid, slabel)    [ST_##group##_##sid] = slabel,
#define DASHBOARD_ERROR(id, code, msg)
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// Emit formatted state: "ST_<GROUP>=<STATE>\r\n" (e.g., "ST_DRIVE=GO\r\n")
static inline void dashboard_emit_state(DashboardStateId st)
{
    char buf[64];

    DashboardStateGroup grp = gStateToGroup[st];
    const char* gname = gStateGroupNames[grp];
    const char* sname = gStateNames[st];
    if (!gname || !sname) return;
    int n = snprintf(buf, sizeof buf, "ST_%s=%s\r\n", gname, sname);
    if (n > 0) dashboard_write_bytes((const uint8_t*)buf, (size_t)n);
}


// --------------------- ERRORS: enum & tables -------------------
#define DASHBOARD_MAPPING(name,off,scale)      /* no-op */
#define DASHBOARD_SIGNAL(id,label,map)         /* no-op */
#define DASHBOARD_STATE_GROUP(group, glabel)   /* no-op */
#define DASHBOARD_STATE(group, sid, slabel)    /* no-op */
#define DASHBOARD_ERROR(id, code, msg)         id,

    typedef enum {
#include "dashboard_signals.def"
        DASHBOARD_ERROR_COUNT
    } DashboardErrorId;

#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR

static const int gErrorCodes[DASHBOARD_ERROR_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)
#define DASHBOARD_STATE(group, sid, slabel)
#define DASHBOARD_ERROR(id, code, msg) [id] = code,
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

static const char* const gErrorNames[DASHBOARD_ERROR_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)
#define DASHBOARD_STATE(group, sid, slabel)
#define DASHBOARD_ERROR(id, code, msg) [id] = #id,
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

static const char* const gErrorMessages[DASHBOARD_ERROR_COUNT] = {
#define DASHBOARD_MAPPING(name,off,scale)
#define DASHBOARD_SIGNAL(id,label,map)
#define DASHBOARD_STATE_GROUP(group, glabel)
#define DASHBOARD_STATE(group, sid, slabel)
#define DASHBOARD_ERROR(id, code, msg) [id] = msg,
#include "dashboard_signals.def"
#undef DASHBOARD_MAPPING
#undef DASHBOARD_SIGNAL
#undef DASHBOARD_STATE_GROUP
#undef DASHBOARD_STATE
#undef DASHBOARD_ERROR
};

// Helper to emit errors: "ERR=E_OVERTEMP\r\n" (or include code/message if you want)
static inline void dashboard_emit_error(DashboardErrorId err)
{
    char buf[64];
    int n = snprintf(buf, sizeof buf, "ERR=%s\r\n", gErrorNames[err]);
    if (n > 0) dashboard_write_bytes((const uint8_t*)buf, (size_t)n);

}


#ifdef __cplusplus
} // extern "C"
#endif

